/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/

#include "rename.h"

#include "curl_setup.h"

#if (!defined(CURL_DISABLE_HTTP) || !defined(CURL_DISABLE_COOKIES)) || \
  !defined(CURL_DISABLE_ALTSVC)

#include "curlx/multibyte.h"
#include "curlx/timeval.h"

/* return 0 on success, 1 on error */
int Curl_rename(const char *oldpath, const char *newpath)
{
#ifdef _WIN32
  /* rename() on Windows does not overwrite, so we cannot use it here.
     MoveFileEx() will overwrite and is usually atomic, however it fails
     when there are open handles to the file. */
  const int max_wait_ms = 1000;
  struct curltime start = curlx_now();
  TCHAR *tchar_oldpath = curlx_convert_UTF8_to_tchar(oldpath);
  TCHAR *tchar_newpath = curlx_convert_UTF8_to_tchar(newpath);
  for(;;) {
    timediff_t diff;
    if(MoveFileEx(tchar_oldpath, tchar_newpath, MOVEFILE_REPLACE_EXISTING)) {
      curlx_free(tchar_oldpath);
      curlx_free(tchar_newpath);
      break;
    }
    diff = curlx_timediff_ms(curlx_now(), start);
    if(diff < 0 || diff > max_wait_ms) {
      curlx_free(tchar_oldpath);
      curlx_free(tchar_newpath);
      return 1;
    }
    Sleep(1);
  }
#else
  if(rename(oldpath, newpath))
    return 1;
#endif
  return 0;
}

#endif
