/*
 *  $Id: selection-range.c 28437 2025-08-24 15:57:54Z yeti-dn $
 *  Copyright (C) 2006-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"

#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"
#include "libgwyddion/serialize.h"
#include "libgwyddion/serializable-utils.h"
#include "libgwyddion/types.h"

#include "libgwyddion/selection-range.h"

#define TYPE_NAME "GwySelectionRange"

#define DEFAULT_ORIENTATION GWY_ORIENTATION_HORIZONTAL

enum {
    OBJECT_SIZE = 2
};

enum {
    PROP_0,
    PROP_ORIENTATION,
    NUM_PROPERTIES,
};

enum {
    ITEM_ORIENTATION,
    NUM_ITEMS
};

struct _GwySelectionRangePrivate {
    GwyOrientation orientation;
};

static void             serializable_init     (GwySerializableInterface *iface);
static void             serializable_itemize  (GwySerializable *serializable,
                                               GwySerializableGroup *group);
static gboolean         serializable_construct(GwySerializable *serializable,
                                               GwySerializableGroup *group,
                                               GwyErrorList **error_list);
static GwySerializable* serializable_copy     (GwySerializable *serializable);
static void             serializable_assign   (GwySerializable *destination,
                                               GwySerializable *source);
static void             set_property          (GObject *object,
                                               guint prop_id,
                                               const GValue *value,
                                               GParamSpec *pspec);
static void             get_property          (GObject *object,
                                               guint prop_id,
                                               GValue *value,
                                               GParamSpec *pspec);
static gboolean         crop_object           (GwySelection *selection,
                                               gint i,
                                               gpointer user_data);
static void             crop                  (GwySelection *selection,
                                               gdouble xmin,
                                               gdouble ymin,
                                               gdouble xmax,
                                               gdouble ymax);
static void             move                  (GwySelection *selection,
                                               gdouble vx,
                                               gdouble vy);
static const gchar*     coord_symbol          (GwySelection *selection,
                                               gint i);

static GParamSpec *properties[NUM_PROPERTIES] = { NULL, };
static GObjectClass *parent_class = NULL;
static GwySerializableInterface *serializable_parent_iface = NULL;

static GwySerializableItem serializable_items[NUM_ITEMS] = {
    { .name = "orientation", .ctype = GWY_SERIALIZABLE_INT32, .value.v_int32 = DEFAULT_ORIENTATION },
};

G_DEFINE_TYPE_WITH_CODE(GwySelectionRange, gwy_selection_range, GWY_TYPE_SELECTION,
                        G_ADD_PRIVATE(GwySelectionRange)
                        G_IMPLEMENT_INTERFACE(GWY_TYPE_SERIALIZABLE, serializable_init))

static void
define_properties(void)
{
    if (properties[PROP_ORIENTATION])
        return;

    /**
     * GwySelectionRange:orientation:
     *
     * The :orientation property represents the orientation of the selected area.
     *
     * The orientation is %GWY_ORIENTATION_HORIZONTAL for selections of ranges along the @x-axis, and
     * %GWY_ORIENTATION_VERTICAL for selections along the @y-axis.
     **/
    properties[PROP_ORIENTATION] = g_param_spec_enum("orientation", NULL,
                                                     "Along which directions the intervals are oriented",
                                                     GWY_TYPE_ORIENTATION, DEFAULT_ORIENTATION, GWY_GPARAM_RWE);
}

static void
gwy_selection_range_class_init(GwySelectionRangeClass *klass)
{
    GwySelectionClass *sel_class = GWY_SELECTION_CLASS(klass);
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

    parent_class = gwy_selection_range_parent_class;

    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;

    sel_class->object_size = OBJECT_SIZE;
    sel_class->crop = crop;
    sel_class->move = move;
    sel_class->coord_symbol = coord_symbol;

    define_properties();
    g_object_class_install_properties(gobject_class, NUM_PROPERTIES, properties);
}

static void
serializable_init(GwySerializableInterface *iface)
{
    serializable_parent_iface = g_type_interface_peek_parent(iface);

    iface->itemize   = serializable_itemize;
    iface->construct = serializable_construct;
    iface->copy      = serializable_copy;
    iface->assign    = serializable_assign;

    define_properties();
    serializable_items[ITEM_ORIENTATION].aux.pspec = properties[PROP_ORIENTATION];
    gwy_fill_serializable_defaults_pspec(serializable_items, NUM_ITEMS, FALSE);
}

static void
gwy_selection_range_init(GwySelectionRange *selection)
{
    GwySelectionRangePrivate *priv;

    priv = selection->priv = gwy_selection_range_get_instance_private(selection);
    priv->orientation = DEFAULT_ORIENTATION;
}

static void
set_property(GObject *object,
             guint prop_id,
             const GValue *value,
             GParamSpec *pspec)
{
    GwySelectionRange *selection = GWY_SELECTION_RANGE(object);

    switch (prop_id) {
        case PROP_ORIENTATION:
        gwy_selection_range_set_orientation(selection, g_value_get_enum(value));
        break;

        default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
get_property(GObject *object,
             guint prop_id,
             GValue *value,
             GParamSpec *pspec)
{
    GwySelectionRangePrivate *priv = GWY_SELECTION_RANGE(object)->priv;

    switch (prop_id) {
        case PROP_ORIENTATION:
        g_value_set_enum(value, priv->orientation);
        break;

        default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static gboolean
crop_object(GwySelection *selection, gint i, gpointer user_data)
{
    GwySelectionRangePrivate *priv = GWY_SELECTION_RANGE(selection)->priv;
    const gdouble *minmax = (const gdouble*)user_data;
    gdouble xy[OBJECT_SIZE];

    gwy_selection_get_object(selection, i, xy);
    if (priv->orientation == GWY_ORIENTATION_VERTICAL)
        return xy[0] >= minmax[0] && xy[0] <= minmax[2] && xy[1] >= minmax[0] && xy[1] <= minmax[2];
    else
        return xy[0] >= minmax[1] && xy[0] <= minmax[3] && xy[1] >= minmax[1] && xy[1] <= minmax[3];
}

static void
crop(GwySelection *selection, gdouble xmin, gdouble ymin, gdouble xmax, gdouble ymax)
{
    gdouble minmax[4] = { xmin, ymin, xmax, ymax };

    gwy_selection_filter(selection, crop_object, minmax);
}

static void
move(GwySelection *selection, gdouble vx, gdouble vy)
{
    GwySelectionRangePrivate *priv = GWY_SELECTION_RANGE(selection)->priv;
    gdouble v[OBJECT_SIZE] = { priv->orientation == GWY_ORIENTATION_VERTICAL ? vy : vx };
    gwy_selection_move_objects(selection, v);
}

static const gchar*
coord_symbol(GwySelection *selection, gint i)
{
    GwySelectionRangePrivate *priv = GWY_SELECTION_RANGE(selection)->priv;
    if (priv->orientation == GWY_ORIENTATION_VERTICAL)
        return i ? "y2" : "y1";
    else
        return i ? "x2" : "x1";
}


/**
 * gwy_selection_range_set_orientation:
 * @selection: A range selection.
 * @orientation: New orientation.
 *
 * Sets the orientation of a range selection.
 *
 * Horizontal orientation means intervals along the X-axis. Vertical orientation means intervals along the Y-axis.
 *
 * Returns: A new selection object.
 **/
void
gwy_selection_range_set_orientation(GwySelectionRange *selection,
                                    GwyOrientation orientation)
{
    g_return_if_fail(GWY_IS_SELECTION_RANGE(selection));
    g_return_if_fail(orientation == GWY_ORIENTATION_HORIZONTAL || orientation == GWY_ORIENTATION_VERTICAL);

    GwySelectionRangePrivate *priv = selection->priv;
    if (orientation == priv->orientation)
        return;

    gwy_selection_clear(GWY_SELECTION(selection));
    priv->orientation = orientation;
    g_object_notify_by_pspec(G_OBJECT(selection), properties[PROP_ORIENTATION]);
}

/**
 * gwy_selection_range_get_orientation:
 * @selection: A range selection.
 *
 * Obtains the orientation of a range selection.
 *
 * Horizontal orientation means intervals along the X-axis. Vertical orientation means intervals along the Y-axis.
 *
 * Returns: The orientation.
 **/
GwyOrientation
gwy_selection_range_get_orientation(GwySelectionRange *selection)
{
    g_return_val_if_fail(GWY_IS_SELECTION_RANGE(selection), DEFAULT_ORIENTATION);

    return selection->priv->orientation;
}

/**
 * gwy_selection_range_new:
 *
 * Creates a new range-wise selection.
 *
 * Returns: A new selection object.
 **/
GwySelection*
gwy_selection_range_new(void)
{
    return (GwySelection*)g_object_new(GWY_TYPE_SELECTION_RANGE, NULL);
}

static void
serializable_itemize(GwySerializable *serializable, GwySerializableGroup *group)
{
    GwySelectionRange *selection = GWY_SELECTION_RANGE(serializable);
    GwySelectionRangePrivate *priv = selection->priv;

    serializable_parent_iface->itemize(serializable, group);
    gwy_serializable_group_append_int32(group, serializable_items + ITEM_ORIENTATION, priv->orientation);
}

static gboolean
serializable_construct(GwySerializable *serializable, GwySerializableGroup *group, GwyErrorList **error_list)
{
    if (!serializable_parent_iface->construct(serializable, group, error_list))
        return FALSE;

    GwySerializableItem its[NUM_ITEMS];
    gwy_assign(its, serializable_items, NUM_ITEMS);
    gwy_deserialize_filter_items(its, NUM_ITEMS, group, TYPE_NAME, error_list);

    GwySelectionRange *selection = GWY_SELECTION_RANGE(serializable);
    selection->priv->orientation = its[ITEM_ORIENTATION].value.v_int32;

    return TRUE;
}

static GwySerializable*
serializable_copy(GwySerializable *serializable)
{
    GwySelectionRange *selection = GWY_SELECTION_RANGE(serializable);
    GwySelectionRange *copy = GWY_SELECTION_RANGE(serializable_parent_iface->copy(serializable));
    copy->priv->orientation = selection->priv->orientation;
    return GWY_SERIALIZABLE(copy);
}

static void
serializable_assign(GwySerializable *destination, GwySerializable *source)
{
    GwySelectionRange *destselection = GWY_SELECTION_RANGE(destination);
    GwySelectionRange *srcselection = GWY_SELECTION_RANGE(source);
    destselection->priv->orientation = srcselection->priv->orientation;
    serializable_parent_iface->assign(destination, source);
}

/**
 * SECTION:selection-range
 * @title: GwySelectionRange
 * @short_description: Interval selection
 *
 * #GwySelectionRange is used to represent horizontal or vertical intervals. Selection data consists of
 * coordinate pairs (from, to).
 *
 * If you obtain the selection from a graph widget it has the "orientation" property set for information.  The
 * orientation has to kept intact in this case.  The graph keeps two distinct range selection objects, one horizontal
 * and one vertical.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
