/**
 ******************************************************************************
 * @defgroup    FIDOmemmap.h  Memory map include file for Fido
 ****************************************************************************@{
 *
 *  @par COPYRIGHT:
 *    COPYRIGHT 2005-2006 INNOVASIC SEMICONDUCTOR, ALL RIGHTS RESERVED.\n
 *    Part of the FIDO REALTIME SUPPORT LIBRARY
 *
 *  @par Created:
 *    Wed, 1 June 2005 David L. Deis [DD]
 *
 *  @par CVS-Info:
 *    $Revision$
 *    $Author$
 *    $Date$
 *
 *  @par Description:
 *  Contains board-specific and onchip memory-mapped register and internal
 *  memory definitions.
 */
#ifndef FIDOmemmap_h
#define FIDOmemmap_h

/******************************************************************************
 * Board-specific definitions specify the location and size of all external
 * memories and the board operating frequency. These values must be specified
 * based upon the physical board design. These definitions are used by the
 * software libraries. 
 ******************************************************************************
 */
 
/**
 * External memory base addresses controlled by chip selects 0,1, and 2.
 * Specify -1 for any absent memories. The defaults are for the Innovasic
 * development board.
 */
#define FLASH_BASE_ADDR             0x00000000
#define FLASH_SIZE                  (8 * 1024 * 1024L)

#define SDRAM_BASE_ADDR             0x02000000
#define SDRAM_SIZE                  (8 * 1024 * 1024L)

#define SRAM_BASE_ADDR              0x03000000  
#define SRAM_SIZE                   (1 * 1024 * 1024L)

/**
 * System clock frequency 
 */
#define FIDO_CLOCK_FREQUENCY         (66 * (1000 * 1000))

/******************************************************************************
 * Onchip memory-mapped register and internal memory definitions.
 ******************************************************************************
 */
 
/**
 * Memory offset register value loaded during crt0. This offset applies to 
 * the onchip SRAM, cache, and frame buffer memories and all memory-mapped 
 * registers and cannot conflict with any external memory regions.
 */
#define FIDO_MEM_OFFSET              0x01000000

/**
 * Onchip SRAM 
 */
#define FIDO_SRAM_BASE_ADDR          (0x00000000 + FIDO_MEM_OFFSET)
#define FIDO_SRAM_SIZE               ((6 * 1024) * sizeof (unsigned long))

/**
 * Onchip relocatable cache RAM 
 */
#define FIDO_CACHE_BASE_ADDR         (0x00080000 + FIDO_MEM_OFFSET)
#define FIDO_CACHE_BLOCK_SIZE        (2 * 1024)
#define FIDO_CACHE_BLOCK_MAX         16
#define FIDO_CACHE_SIZE              (FIDO_CACHE_BLOCK_SIZE * FIDO_CACHE_BLOCK_MAX)
#define FIDO_CACHE_BLOCK_ENABLE      (1 << 0)

/**
 * Memory-mapped registers
 */
#ifndef __ASSEMBLER__
    typedef unsigned long volatile FIDO_Register_t;
    #define FIDO_REGISTER_DEF (FIDO_Register_t*)
#else
    #define FIDO_REGISTER_DEF
#endif /* __ASSEMBLER__ */

/* CPU DMA channel 0 and channel 1 registers */
#define FIDO_DMACH0_CONTROL          FIDO_REGISTER_DEF(0xA0000 + FIDO_MEM_OFFSET)
#define FIDO_DMACH0_COUNT            FIDO_REGISTER_DEF(0xA0004 + FIDO_MEM_OFFSET)
#define FIDO_DMACH0_DEST             FIDO_REGISTER_DEF(0xA0008 + FIDO_MEM_OFFSET)
#define FIDO_DMACH0_SOURCE           FIDO_REGISTER_DEF(0xA000C + FIDO_MEM_OFFSET)
#define FIDO_DMACH1_CONTROL          FIDO_REGISTER_DEF(0xA0010 + FIDO_MEM_OFFSET)
#define FIDO_DMACH1_COUNT            FIDO_REGISTER_DEF(0xA0014 + FIDO_MEM_OFFSET)
#define FIDO_DMACH1_DEST             FIDO_REGISTER_DEF(0xA0018 + FIDO_MEM_OFFSET)
#define FIDO_DMACH1_SOURCE           FIDO_REGISTER_DEF(0xA001C + FIDO_MEM_OFFSET)

#define FIDO_DMACH_MAX               2

/* Timer-Counter unit 0 registers */
#define FIDO_TCU00_STATUS            FIDO_REGISTER_DEF(0xA0280 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_MODE              FIDO_REGISTER_DEF(0xA0284 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_COUNTER           FIDO_REGISTER_DEF(0xA0288 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH0_IOMODE        FIDO_REGISTER_DEF(0xA0290 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH1_IOMODE        FIDO_REGISTER_DEF(0xA0294 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH2_IOMODE        FIDO_REGISTER_DEF(0xA0298 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH3_IOMODE        FIDO_REGISTER_DEF(0xA029C + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH0_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02A0 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH1_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02A4 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH2_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02A8 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH3_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02AC + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH0_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02B0 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH1_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02B4 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH2_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02B8 + FIDO_MEM_OFFSET)
#define FIDO_TCU00_CH3_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02BC + FIDO_MEM_OFFSET)

/* Timer-Counter unit 1 registers */
#define FIDO_TCU01_STATUS            FIDO_REGISTER_DEF(0xA02C0 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_MODE              FIDO_REGISTER_DEF(0xA02C4 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_COUNTER           FIDO_REGISTER_DEF(0xA02C8 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH0_IOMODE        FIDO_REGISTER_DEF(0xA02D0 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH1_IOMODE        FIDO_REGISTER_DEF(0xA02D4 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH2_IOMODE        FIDO_REGISTER_DEF(0xA02D8 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH3_IOMODE        FIDO_REGISTER_DEF(0xA02DC + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH0_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02E0 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH1_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02E4 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH2_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02E8 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH3_INPUTCAPTURE  FIDO_REGISTER_DEF(0xA02EC + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH0_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02F0 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH1_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02F4 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH2_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02F8 + FIDO_MEM_OFFSET)
#define FIDO_TCU01_CH3_OUTPUTCOMPARE FIDO_REGISTER_DEF(0xA02FC + FIDO_MEM_OFFSET)

/* SystemTimer interrupt control registers, main ctrl reg, and prescale */
#define FIDO_SYSTIMER_INT00CONTROL   FIDO_REGISTER_DEF(0xA0300 + FIDO_MEM_OFFSET)
#define FIDO_SYSTIMER_INT01CONTROL   FIDO_REGISTER_DEF(0xA0304 + FIDO_MEM_OFFSET)
#define FIDO_SYSTIMER_INT02CONTROL   FIDO_REGISTER_DEF(0xA0308 + FIDO_MEM_OFFSET)
#define FIDO_SYSTIMER_INT03CONTROL   FIDO_REGISTER_DEF(0xA030C + FIDO_MEM_OFFSET)
#define FIDO_SYSTIMER_INT04CONTROL   FIDO_REGISTER_DEF(0xA0310 + FIDO_MEM_OFFSET)
#define FIDO_SYSTIMER_CONTROL        FIDO_REGISTER_DEF(0xA0314 + FIDO_MEM_OFFSET)
#define FIDO_SYSTIMER_PRESCALE       FIDO_REGISTER_DEF(0xA0318 + FIDO_MEM_OFFSET)

/* Watchdog timer control and reload registers */
#define FIDO_WDTTIMER_CONTROL        FIDO_REGISTER_DEF(0xA0340 + FIDO_MEM_OFFSET)
#define FIDO_WDTTIMER_RELOAD         FIDO_REGISTER_DEF(0xA0344 + FIDO_MEM_OFFSET)

#define FIDO_WDTTIMER_CONTROL_RELOAD    (1 << 0)
#define FIDO_WDTTIMER_CONTROL_ENABLE    (1 << 1)
#define FIDO_WDTTIMER_CONTROL_INTENABLE (1 << 2)

/*  Power on reset register */
#define FIDO_POR_REG                 FIDO_REGISTER_DEF(0xA0360 + FIDO_MEM_OFFSET)

/*  Clock mask register, fixed at 0x00003F0F */
#define FIDO_CLOCK_MASK_REGISTER     FIDO_REGISTER_DEF(0xA0380 + FIDO_MEM_OFFSET)
#define FIDO_CLOCK_MASK_TCU1         (unsigned long)(1 << 13)
#define FIDO_CLOCK_MASK_TCU0         (unsigned long)(1 << 12)
#define FIDO_CLOCK_MASK_SYSTIMER     (unsigned long)(1 << 11)
#define FIDO_CLOCK_MASK_SDRAM        (unsigned long)(1 << 10)
#define FIDO_CLOCK_MASK_ATOD         (unsigned long)(1 <<  9)
#define FIDO_CLOCK_MASK_PMU          (unsigned long)(1 <<  8)
#define FIDO_CLOCK_MASK_UIC3         (unsigned long)(1 <<  3)
#define FIDO_CLOCK_MASK_UIC2         (unsigned long)(1 <<  2)
#define FIDO_CLOCK_MASK_UIC1         (unsigned long)(1 <<  1)
#define FIDO_CLOCK_MASK_UIC0         (unsigned long)(1 <<  0)

/*  Device ID register, fixed at 0x29811000 */
#define FIDO_DEVICE_ID_REGISTER      FIDO_REGISTER_DEF(0xA0400 + FIDO_MEM_OFFSET)

/*  Debug control registers for use by GDB and ROM monitor */
#define FIDO_DBG_CTRL                FIDO_REGISTER_DEF(0xA0404 + FIDO_MEM_OFFSET)
#define FIDO_TRCBUF_CTRL             FIDO_REGISTER_DEF(0xA0408 + FIDO_MEM_OFFSET)
#define FIDO_TRCBUF_BASE             FIDO_REGISTER_DEF(0xA040C + FIDO_MEM_OFFSET)
#define FIDO_BRK00_BASE              FIDO_REGISTER_DEF(0xA0410 + FIDO_MEM_OFFSET)
#define FIDO_BRK00_DATA              FIDO_REGISTER_DEF(0xA0414 + FIDO_MEM_OFFSET)
#define FIDO_BRK00_DATAMASK          FIDO_REGISTER_DEF(0xA0418 + FIDO_MEM_OFFSET)
#define FIDO_BRK00_CTRL              FIDO_REGISTER_DEF(0xA041C + FIDO_MEM_OFFSET)
#define FIDO_BRK01_BASE              FIDO_REGISTER_DEF(0xA0420 + FIDO_MEM_OFFSET)
#define FIDO_BRK01_DATA              FIDO_REGISTER_DEF(0xA0424 + FIDO_MEM_OFFSET)
#define FIDO_BRK01_DATAMASK          FIDO_REGISTER_DEF(0xA0428 + FIDO_MEM_OFFSET)
#define FIDO_BRK01_CTRL              FIDO_REGISTER_DEF(0xA042C + FIDO_MEM_OFFSET)
#define FIDO_BRK02_BASE              FIDO_REGISTER_DEF(0xA0430 + FIDO_MEM_OFFSET)
#define FIDO_BRK02_DATA              FIDO_REGISTER_DEF(0xA0434 + FIDO_MEM_OFFSET)
#define FIDO_BRK02_DATAMASK          FIDO_REGISTER_DEF(0xA0438 + FIDO_MEM_OFFSET)
#define FIDO_BRK02_CTRL              FIDO_REGISTER_DEF(0xA043C + FIDO_MEM_OFFSET)
#define FIDO_BRK03_BASE              FIDO_REGISTER_DEF(0xA0440 + FIDO_MEM_OFFSET)
#define FIDO_BRK03_DATA              FIDO_REGISTER_DEF(0xA0444 + FIDO_MEM_OFFSET)
#define FIDO_BRK03_DATAMASK          FIDO_REGISTER_DEF(0xA0448 + FIDO_MEM_OFFSET)
#define FIDO_BRK03_CTRL              FIDO_REGISTER_DEF(0xA044C + FIDO_MEM_OFFSET)
#define FIDO_BRK04_BASE              FIDO_REGISTER_DEF(0xA0450 + FIDO_MEM_OFFSET)
#define FIDO_BRK04_DATA              FIDO_REGISTER_DEF(0xA0454 + FIDO_MEM_OFFSET)
#define FIDO_BRK04_DATAMASK          FIDO_REGISTER_DEF(0xA0458 + FIDO_MEM_OFFSET)
#define FIDO_BRK04_CTRL              FIDO_REGISTER_DEF(0xA045C + FIDO_MEM_OFFSET)
#define FIDO_BRK05_BASE              FIDO_REGISTER_DEF(0xA0460 + FIDO_MEM_OFFSET)
#define FIDO_BRK05_DATA              FIDO_REGISTER_DEF(0xA0464 + FIDO_MEM_OFFSET)
#define FIDO_BRK05_DATAMASK          FIDO_REGISTER_DEF(0xA0468 + FIDO_MEM_OFFSET)
#define FIDO_BRK05_CTRL              FIDO_REGISTER_DEF(0xA046C + FIDO_MEM_OFFSET)
#define FIDO_BRK06_BASE              FIDO_REGISTER_DEF(0xA0470 + FIDO_MEM_OFFSET)
#define FIDO_BRK06_DATA              FIDO_REGISTER_DEF(0xA0474 + FIDO_MEM_OFFSET)
#define FIDO_BRK06_DATAMASK          FIDO_REGISTER_DEF(0xA0478 + FIDO_MEM_OFFSET)
#define FIDO_BRK06_CTRL              FIDO_REGISTER_DEF(0xA047C + FIDO_MEM_OFFSET)
#define FIDO_BRK07_BASE              FIDO_REGISTER_DEF(0xA0480 + FIDO_MEM_OFFSET)
#define FIDO_BRK07_DATA              FIDO_REGISTER_DEF(0xA0484 + FIDO_MEM_OFFSET)
#define FIDO_BRK07_DATAMASK          FIDO_REGISTER_DEF(0xA0488 + FIDO_MEM_OFFSET)
#define FIDO_BRK07_CTRL              FIDO_REGISTER_DEF(0xA048C + FIDO_MEM_OFFSET)

/*  A/D  registers */
#define FIDO_A2D0_REG00              FIDO_REGISTER_DEF(0xA0600 + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG01              FIDO_REGISTER_DEF(0xA0604 + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG02              FIDO_REGISTER_DEF(0xA0608 + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG03              FIDO_REGISTER_DEF(0xA060C + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG04              FIDO_REGISTER_DEF(0xA0610 + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG05              FIDO_REGISTER_DEF(0xA0614 + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG06              FIDO_REGISTER_DEF(0xA0618 + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG07              FIDO_REGISTER_DEF(0xA061C + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG08              FIDO_REGISTER_DEF(0xA0620 + FIDO_MEM_OFFSET)
#define FIDO_A2D0_REG09              FIDO_REGISTER_DEF(0xA0624 + FIDO_MEM_OFFSET)

/*  Chip select control&timing register pairs */
#define FIDO_BIU_CS0_CONTROL         FIDO_REGISTER_DEF(0xA0680 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS0_TIMING          FIDO_REGISTER_DEF(0xA0684 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS1_CONTROL         FIDO_REGISTER_DEF(0xA0688 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS1_TIMING          FIDO_REGISTER_DEF(0xA068C + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS2_CONTROL         FIDO_REGISTER_DEF(0xA0690 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS2_TIMING          FIDO_REGISTER_DEF(0xA0694 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS3_CONTROL         FIDO_REGISTER_DEF(0xA0698 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS3_TIMING          FIDO_REGISTER_DEF(0xA069C + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS4_CONTROL         FIDO_REGISTER_DEF(0xA06A0 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS4_TIMING          FIDO_REGISTER_DEF(0xA06A4 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS5_CONTROL         FIDO_REGISTER_DEF(0xA06A8 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS5_TIMING          FIDO_REGISTER_DEF(0xA06AC + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS6_CONTROL         FIDO_REGISTER_DEF(0xA06B0 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS6_TIMING          FIDO_REGISTER_DEF(0xA06B4 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS7_CONTROL         FIDO_REGISTER_DEF(0xA06B8 + FIDO_MEM_OFFSET)
#define FIDO_BIU_CS7_TIMING          FIDO_REGISTER_DEF(0xA06BC + FIDO_MEM_OFFSET)

#define FIDO_BIU_PRIORITY            FIDO_REGISTER_DEF(0xA0700 + FIDO_MEM_OFFSET)
#define FIDO_BIU_DEF_TIMING          FIDO_REGISTER_DEF(0xA0704 + FIDO_MEM_OFFSET)

/*  SDRAM timing, control, and bank select registers */
#define FIDO_SDRAM_TIMING_0          FIDO_REGISTER_DEF(0xA0800 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_TIMING_1          FIDO_REGISTER_DEF(0xA0804 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_CONFIG_0          FIDO_REGISTER_DEF(0xA0808 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_CONFIG_1          FIDO_REGISTER_DEF(0xA080C + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_0        FIDO_REGISTER_DEF(0xA0810 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_1        FIDO_REGISTER_DEF(0xA0814 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_2        FIDO_REGISTER_DEF(0xA0818 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_3        FIDO_REGISTER_DEF(0xA081C + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_4        FIDO_REGISTER_DEF(0xA0820 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_5        FIDO_REGISTER_DEF(0xA0824 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_6        FIDO_REGISTER_DEF(0xA0828 + FIDO_MEM_OFFSET)
#define FIDO_SDRAM_EXT_BANK_7        FIDO_REGISTER_DEF(0xA082C + FIDO_MEM_OFFSET)

/*  External interrupt control registers */
#define FIDO_INTCONTROLCH0           FIDO_REGISTER_DEF(0xA0900 + FIDO_MEM_OFFSET)
#define FIDO_INTCONTROLCH1           FIDO_REGISTER_DEF(0xA0904 + FIDO_MEM_OFFSET)
#define FIDO_INTCONTROLCH2           FIDO_REGISTER_DEF(0xA0908 + FIDO_MEM_OFFSET)
#define FIDO_INTCONTROLCH3           FIDO_REGISTER_DEF(0xA090C + FIDO_MEM_OFFSET)
#define FIDO_INTCONTROLCH4           FIDO_REGISTER_DEF(0xA0910 + FIDO_MEM_OFFSET)
#define FIDO_INTCONTROLCH5           FIDO_REGISTER_DEF(0xA0914 + FIDO_MEM_OFFSET)
#define FIDO_INTCONTROLCH6           FIDO_REGISTER_DEF(0xA0918 + FIDO_MEM_OFFSET)
#define FIDO_INTCONTROLCH7           FIDO_REGISTER_DEF(0xA091C + FIDO_MEM_OFFSET)

/* Software interrupt control registers */
#define FIDO_CTX0_INT_CTRL           FIDO_REGISTER_DEF(0xA0980 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_INT_CTRL           FIDO_REGISTER_DEF(0xA0984 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_INT_CTRL           FIDO_REGISTER_DEF(0xA0988 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_INT_CTRL           FIDO_REGISTER_DEF(0xA098C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_INT_CTRL           FIDO_REGISTER_DEF(0xA0990 + FIDO_MEM_OFFSET)

#define FIDO_PMU_MAC_FILTER_MODE     FIDO_REGISTER_DEF(0xA0A00 + FIDO_MEM_OFFSET)
#define FIDO_PMU_FILTER_DATA_HEAD    FIDO_REGISTER_DEF(0xA0A04 + FIDO_MEM_OFFSET)
#define FIDO_PMU_FILTER_RDDATA_HEAD  FIDO_REGISTER_DEF(0xA0A08 + FIDO_MEM_OFFSET)

/* PMU channel 0 (tied to UIC 0) registers */
#define FIDO_PMUCH0A_CONTROL         FIDO_REGISTER_DEF(0xA0A40 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_STATUS          FIDO_REGISTER_DEF(0xA0A44 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0A48 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0A4C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0A50 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0A54 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0A58 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0A5C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0A60 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0A64 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0A68 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0A6C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_XMITDATA        FIDO_REGISTER_DEF(0xA0A70 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0A_RCVDATA         FIDO_REGISTER_DEF(0xA0A74 + FIDO_MEM_OFFSET)

#define FIDO_PMUCH0B_CONTROL         FIDO_REGISTER_DEF(0xA0A80 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_STATUS          FIDO_REGISTER_DEF(0xA0A84 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0A88 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0A8C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0A90 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0A94 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0A98 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0A9C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0AA0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0AA4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0AA8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0AAC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_XMITDATA        FIDO_REGISTER_DEF(0xA0AB0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH0B_RCVDATA         FIDO_REGISTER_DEF(0xA0AB4 + FIDO_MEM_OFFSET)

/* PMU channel 1 (tied to UIC 1) registers */
#define FIDO_PMUCH1A_CONTROL         FIDO_REGISTER_DEF(0xA0AC0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_STATUS          FIDO_REGISTER_DEF(0xA0AC4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0AC8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0ACC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0AD0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0AD4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0AD8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0ADC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0AE0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0AE4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0AE8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0AEC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_XMITDATA        FIDO_REGISTER_DEF(0xA0AF0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1A_RCVDATA         FIDO_REGISTER_DEF(0xA0AF4 + FIDO_MEM_OFFSET)

#define FIDO_PMUCH1B_CONTROL         FIDO_REGISTER_DEF(0xA0B00 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_STATUS          FIDO_REGISTER_DEF(0xA0B04 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0B08 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0B0C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0B10 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0B14 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0B18 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0B1C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0B20 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0B24 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0B28 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0B2C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_XMITDATA        FIDO_REGISTER_DEF(0xA0B30 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH1B_RCVDATA         FIDO_REGISTER_DEF(0xA0B34 + FIDO_MEM_OFFSET)

/* PMU channel 2 (tied to UIC 2) registers */
#define FIDO_PMUCH2A_CONTROL         FIDO_REGISTER_DEF(0xA0B40 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_STATUS          FIDO_REGISTER_DEF(0xA0B44 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0B48 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0B4C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0B50 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0B54 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0B58 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0B5C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0B60 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0B64 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0B68 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0B6C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_XMITDATA        FIDO_REGISTER_DEF(0xA0B70 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2A_RCVDATA         FIDO_REGISTER_DEF(0xA0B74 + FIDO_MEM_OFFSET)

#define FIDO_PMUCH2B_CONTROL         FIDO_REGISTER_DEF(0xA0B80 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_STATUS          FIDO_REGISTER_DEF(0xA0B84 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0B88 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0B8C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0B90 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0B94 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0B98 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0B9C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0BA0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0BA4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0BA8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0BAC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_XMITDATA        FIDO_REGISTER_DEF(0xA0BB0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH2B_RCVDATA         FIDO_REGISTER_DEF(0xA0BB4 + FIDO_MEM_OFFSET)

/* PMU channel 3 (tied to UIC 3) registers */
#define FIDO_PMUCH3A_CONTROL         FIDO_REGISTER_DEF(0xA0BC0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_STATUS          FIDO_REGISTER_DEF(0xA0BC4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0BC8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0BCC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0BD0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0BD4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0BD8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0BDC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0BE0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0BE4 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0BE8 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0BEC + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_XMITDATA        FIDO_REGISTER_DEF(0xA0BF0 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3A_RCVDATA         FIDO_REGISTER_DEF(0xA0BF4 + FIDO_MEM_OFFSET)

#define FIDO_PMUCH3B_CONTROL         FIDO_REGISTER_DEF(0xA0C00 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_STATUS          FIDO_REGISTER_DEF(0xA0C04 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_PCKXMITSIZE     FIDO_REGISTER_DEF(0xA0C08 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_PCKRCVSIZE      FIDO_REGISTER_DEF(0xA0C0C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_XMITFBUFSTART   FIDO_REGISTER_DEF(0xA0C10 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_XMITFBUFEND     FIDO_REGISTER_DEF(0xA0C14 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_XMITFBUFRDPTR   FIDO_REGISTER_DEF(0xA0C18 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_XMITFBUFWRPTR   FIDO_REGISTER_DEF(0xA0C1C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_RCVFBUFSTART    FIDO_REGISTER_DEF(0xA0C20 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_RCVFBUFEND      FIDO_REGISTER_DEF(0xA0C24 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_RCVFBUFRDPTR    FIDO_REGISTER_DEF(0xA0C28 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_RCVFBUFWRPTR    FIDO_REGISTER_DEF(0xA0C2C + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_XMITDATA        FIDO_REGISTER_DEF(0xA0C30 + FIDO_MEM_OFFSET)
#define FIDO_PMUCH3B_RCVDATA         FIDO_REGISTER_DEF(0xA0C34 + FIDO_MEM_OFFSET)

/* 4 PMU/UICs with 2 UIC channels (A and B) each */
#define FIDO_PMUCH_MAX               4
#define FIDO_UIC_MAX                 FIDO_PMUCH_MAX
#define FIDO_UICCH_MAX               2

/* FIDO_PMUCHXX_Control register definitions */
#define FIDO_PMU_RCV_PACKET_SIZE_EN  ( 1 << 18)
#define FIDO_PMU_UIC_INT_EN          ( 1 << 17)
#define FIDO_PMU_PMU_INT_EN          ( 1 << 16)
#define FIDO_PMU_PRI_SHIFT           13
#define FIDO_PMU_PRI_MASK            ( 7 << FIDO_PMU_PRI_SHIFT)
#define FIDO_PMU_CONTEXT_SHIFT       8
#define FIDO_PMU_CONTEXT_MASK        ( 7 << FIDO_PMU_CONTEXT_SHIFT)
#define FIDO_PMU_CHANNEL_EN          ( 1 <<  7)
#define FIDO_PMU_XMIT_START          ( 1 <<  6)
#define FIDO_PMU_XMIT_IRQ_EN         ( 1 <<  4)
#define FIDO_PMU_RCV_IRQ_EN          ( 1 <<  3)
#define FIDO_PMU_ERROR_IRQ_EN        ( 1 <<  2)
#define FIDO_PMU_MODE_FIFO           ( 0 <<  0)
#define FIDO_PMU_MODE_RANDOM         ( 1 <<  0)
#define FIDO_PMU_MODE_MASK           ( 3 <<  0)

/* FIDO_PMUCHXX_Status register definitions */
#define FIDO_PMU_RCV_FIFO_FULL       ( 1 << 11)
#define FIDO_PMU_RCV_FIFO_EMPTY      ( 1 << 10)
#define FIDO_PMU_XMIT_FIFO_FULL      ( 1 <<  9)
#define FIDO_PMU_XMIT_FIFO_EMPTY     ( 1 <<  8)
#define FIDO_PMU_INTERRUPT           ( 1 <<  7)
#define FIDO_PMU_RCV_BUFF_OVERFLOW   ( 1 <<  3)
#define FIDO_PMU_UIC_ERROR           ( 1 <<  2)
#define FIDO_PMU_RCV_COMPLETE        ( 1 <<  1)
#define FIDO_PMU_XMIT_COMPLETE       ( 1 <<  0)

/* PMU interrupt vector definitions */
#define FIDO_PMU0_INT_VECTOR         77
#define FIDO_PMU1_INT_VECTOR         78
#define FIDO_PMU2_INT_VECTOR         79
#define FIDO_PMU3_INT_VECTOR         80 

/* Context claim and software interrupt registers */
#define FIDO_SWINTACT0               FIDO_REGISTER_DEF(0xA110C + FIDO_MEM_OFFSET)
#define FIDO_CTX1_CLAIM              FIDO_REGISTER_DEF(0xA1110 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_PENDING            FIDO_REGISTER_DEF(0xA1114 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_PRI_INHER          FIDO_REGISTER_DEF(0xA1118 + FIDO_MEM_OFFSET)
#define FIDO_SWINTACT1               FIDO_REGISTER_DEF(0xA111C + FIDO_MEM_OFFSET)
#define FIDO_CTX2_CLAIM              FIDO_REGISTER_DEF(0xA1120 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_PENDING            FIDO_REGISTER_DEF(0xA1124 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_PRI_INHER          FIDO_REGISTER_DEF(0xA1128 + FIDO_MEM_OFFSET)
#define FIDO_SWINTACT2               FIDO_REGISTER_DEF(0xA112C + FIDO_MEM_OFFSET)
#define FIDO_CTX3_CLAIM              FIDO_REGISTER_DEF(0xA1130 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_PENDING            FIDO_REGISTER_DEF(0xA1134 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_PRI_INHER          FIDO_REGISTER_DEF(0xA1138 + FIDO_MEM_OFFSET)
#define FIDO_SWINTACT3               FIDO_REGISTER_DEF(0xA113C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_CLAIM              FIDO_REGISTER_DEF(0xA1140 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_PENDING            FIDO_REGISTER_DEF(0xA1144 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_PRI_INHER          FIDO_REGISTER_DEF(0xA1148 + FIDO_MEM_OFFSET)
#define FIDO_SWINTACT4               FIDO_REGISTER_DEF(0xA114C + FIDO_MEM_OFFSET)

/*  Context0 data registers and address registers */
#define FIDO_CTX0_D0                 FIDO_REGISTER_DEF(0xA8100 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_D1                 FIDO_REGISTER_DEF(0xA8104 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_D2                 FIDO_REGISTER_DEF(0xA8108 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_D3                 FIDO_REGISTER_DEF(0xA810C + FIDO_MEM_OFFSET)
#define FIDO_CTX0_D4                 FIDO_REGISTER_DEF(0xA8110 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_D5                 FIDO_REGISTER_DEF(0xA8114 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_D6                 FIDO_REGISTER_DEF(0xA8118 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_D7                 FIDO_REGISTER_DEF(0xA811C + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A0                 FIDO_REGISTER_DEF(0xA8120 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A1                 FIDO_REGISTER_DEF(0xA8124 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A2                 FIDO_REGISTER_DEF(0xA8128 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A3                 FIDO_REGISTER_DEF(0xA812C + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A4                 FIDO_REGISTER_DEF(0xA8130 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A5                 FIDO_REGISTER_DEF(0xA8134 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A6                 FIDO_REGISTER_DEF(0xA8138 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A7                 FIDO_REGISTER_DEF(0xA813C + FIDO_MEM_OFFSET)
#define FIDO_CTX0_A7S                FIDO_REGISTER_DEF(0xA8140 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_PC                 FIDO_REGISTER_DEF(0xA8144 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_SR                 FIDO_REGISTER_DEF(0xA8148 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_VBR                FIDO_REGISTER_DEF(0xA814C + FIDO_MEM_OFFSET)
#define FIDO_CTX0_SFC                FIDO_REGISTER_DEF(0xA8150 + FIDO_MEM_OFFSET)
#define FIDO_CTX0_DFC                FIDO_REGISTER_DEF(0xA8154 + FIDO_MEM_OFFSET)

/*  Context1 data registers and address registers */
#define FIDO_CTX1_D0                 FIDO_REGISTER_DEF(0xA8180 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_D1                 FIDO_REGISTER_DEF(0xA8184 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_D2                 FIDO_REGISTER_DEF(0xA8188 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_D3                 FIDO_REGISTER_DEF(0xA818C + FIDO_MEM_OFFSET)
#define FIDO_CTX1_D4                 FIDO_REGISTER_DEF(0xA8190 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_D5                 FIDO_REGISTER_DEF(0xA8194 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_D6                 FIDO_REGISTER_DEF(0xA8198 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_D7                 FIDO_REGISTER_DEF(0xA819C + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A0                 FIDO_REGISTER_DEF(0xA81A0 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A1                 FIDO_REGISTER_DEF(0xA81A4 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A2                 FIDO_REGISTER_DEF(0xA81A8 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A3                 FIDO_REGISTER_DEF(0xA81AC + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A4                 FIDO_REGISTER_DEF(0xA81B0 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A5                 FIDO_REGISTER_DEF(0xA81B4 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A6                 FIDO_REGISTER_DEF(0xA81B8 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A7                 FIDO_REGISTER_DEF(0xA81BC + FIDO_MEM_OFFSET)
#define FIDO_CTX1_A7S                FIDO_REGISTER_DEF(0xA81C0 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_PC                 FIDO_REGISTER_DEF(0xA81C4 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_SR                 FIDO_REGISTER_DEF(0xA81C8 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_VBR                FIDO_REGISTER_DEF(0xA81CC + FIDO_MEM_OFFSET)
#define FIDO_CTX1_SFC                FIDO_REGISTER_DEF(0xA81D0 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_DFC                FIDO_REGISTER_DEF(0xA81D4 + FIDO_MEM_OFFSET)

/*  Context2 data registers and address registers */
#define FIDO_CTX2_D0                 FIDO_REGISTER_DEF(0xA8200 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_D1                 FIDO_REGISTER_DEF(0xA8204 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_D2                 FIDO_REGISTER_DEF(0xA8208 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_D3                 FIDO_REGISTER_DEF(0xA820C + FIDO_MEM_OFFSET)
#define FIDO_CTX2_D4                 FIDO_REGISTER_DEF(0xA8210 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_D5                 FIDO_REGISTER_DEF(0xA8214 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_D6                 FIDO_REGISTER_DEF(0xA8218 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_D7                 FIDO_REGISTER_DEF(0xA812C + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A0                 FIDO_REGISTER_DEF(0xA8220 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A1                 FIDO_REGISTER_DEF(0xA8224 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A2                 FIDO_REGISTER_DEF(0xA8228 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A3                 FIDO_REGISTER_DEF(0xA822C + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A4                 FIDO_REGISTER_DEF(0xA8230 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A5                 FIDO_REGISTER_DEF(0xA8234 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A6                 FIDO_REGISTER_DEF(0xA8238 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A7                 FIDO_REGISTER_DEF(0xA823C + FIDO_MEM_OFFSET)
#define FIDO_CTX2_A7S                FIDO_REGISTER_DEF(0xA8240 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_PC                 FIDO_REGISTER_DEF(0xA8244 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_SR                 FIDO_REGISTER_DEF(0xA8248 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_VBR                FIDO_REGISTER_DEF(0xA824C + FIDO_MEM_OFFSET)
#define FIDO_CTX2_SFC                FIDO_REGISTER_DEF(0xA8250 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_DFC                FIDO_REGISTER_DEF(0xA8254 + FIDO_MEM_OFFSET)

/*  Context3 data registers and address registers */
#define FIDO_CTX3_D0                 FIDO_REGISTER_DEF(0xA8280 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_D1                 FIDO_REGISTER_DEF(0xA8284 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_D2                 FIDO_REGISTER_DEF(0xA8288 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_D3                 FIDO_REGISTER_DEF(0xA828C + FIDO_MEM_OFFSET)
#define FIDO_CTX3_D4                 FIDO_REGISTER_DEF(0xA8290 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_D5                 FIDO_REGISTER_DEF(0xA8294 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_D6                 FIDO_REGISTER_DEF(0xA8298 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_D7                 FIDO_REGISTER_DEF(0xA829C + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A0                 FIDO_REGISTER_DEF(0xA82A0 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A1                 FIDO_REGISTER_DEF(0xA82A4 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A2                 FIDO_REGISTER_DEF(0xA82A8 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A3                 FIDO_REGISTER_DEF(0xA82AC + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A4                 FIDO_REGISTER_DEF(0xA82B0 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A5                 FIDO_REGISTER_DEF(0xA82B4 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A6                 FIDO_REGISTER_DEF(0xA82B8 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A7                 FIDO_REGISTER_DEF(0xA82BC + FIDO_MEM_OFFSET)
#define FIDO_CTX3_A7S                FIDO_REGISTER_DEF(0xA82C0 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_PC                 FIDO_REGISTER_DEF(0xA82C4 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_SR                 FIDO_REGISTER_DEF(0xA82C8 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_VBR                FIDO_REGISTER_DEF(0xA82CC + FIDO_MEM_OFFSET)
#define FIDO_CTX3_SFC                FIDO_REGISTER_DEF(0xA82D0 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_DFC                FIDO_REGISTER_DEF(0xA82D4 + FIDO_MEM_OFFSET)

/*  Context4 data registers and address registers */
#define FIDO_CTX4_D0                 FIDO_REGISTER_DEF(0xA8300 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_D1                 FIDO_REGISTER_DEF(0xA8304 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_D2                 FIDO_REGISTER_DEF(0xA8308 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_D3                 FIDO_REGISTER_DEF(0xA830C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_D4                 FIDO_REGISTER_DEF(0xA8310 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_D5                 FIDO_REGISTER_DEF(0xA8314 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_D6                 FIDO_REGISTER_DEF(0xA8318 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_D7                 FIDO_REGISTER_DEF(0xA831C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A0                 FIDO_REGISTER_DEF(0xA8320 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A1                 FIDO_REGISTER_DEF(0xA8324 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A2                 FIDO_REGISTER_DEF(0xA8328 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A3                 FIDO_REGISTER_DEF(0xA832C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A4                 FIDO_REGISTER_DEF(0xA8330 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A5                 FIDO_REGISTER_DEF(0xA8334 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A6                 FIDO_REGISTER_DEF(0xA8338 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A7                 FIDO_REGISTER_DEF(0xA833C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_A7S                FIDO_REGISTER_DEF(0xA8340 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_PC                 FIDO_REGISTER_DEF(0xA8344 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_SR                 FIDO_REGISTER_DEF(0xA8348 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_VBR                FIDO_REGISTER_DEF(0xA834C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_SFC                FIDO_REGISTER_DEF(0xA8350 + FIDO_MEM_OFFSET)
#define FIDO_CTX4_DFC                FIDO_REGISTER_DEF(0xA8354 + FIDO_MEM_OFFSET)

#define FIDO_CTX_MAX                 5

/*  MPU Block control register */
#define FIDO_MPU_BLK00CTRL           FIDO_REGISTER_DEF(0xAA000 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK01CTRL           FIDO_REGISTER_DEF(0xAA004 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK02CTRL           FIDO_REGISTER_DEF(0xAA008 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK03CTRL           FIDO_REGISTER_DEF(0xAA00C + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK04CTRL           FIDO_REGISTER_DEF(0xAA010 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK05CTRL           FIDO_REGISTER_DEF(0xAA014 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK06CTRL           FIDO_REGISTER_DEF(0xAA018 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK07CTRL           FIDO_REGISTER_DEF(0xAA01C + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK08CTRL           FIDO_REGISTER_DEF(0xAA020 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK09CTRL           FIDO_REGISTER_DEF(0xAA024 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK10CTRL           FIDO_REGISTER_DEF(0xAA028 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK11CTRL           FIDO_REGISTER_DEF(0xAA02C + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK12CTRL           FIDO_REGISTER_DEF(0xAA030 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK13CTRL           FIDO_REGISTER_DEF(0xAA034 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK14CTRL           FIDO_REGISTER_DEF(0xAA038 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK15CTRL           FIDO_REGISTER_DEF(0xAA03C + FIDO_MEM_OFFSET)

#define FIDO_MPU_BLOCK_MAX           16
#define FIDO_MPU_BLOCK_BASE_ALIGN    64

/*  MPU Block attribute register */
#define FIDO_MPU_BLK00ATTRIB         FIDO_REGISTER_DEF(0xAA080 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK01ATTRIB         FIDO_REGISTER_DEF(0xAA084 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK02ATTRIB         FIDO_REGISTER_DEF(0xAA088 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK03ATTRIB         FIDO_REGISTER_DEF(0xAA08C + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK04ATTRIB         FIDO_REGISTER_DEF(0xAA090 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK05ATTRIB         FIDO_REGISTER_DEF(0xAA094 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK06ATTRIB         FIDO_REGISTER_DEF(0xAA098 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK07ATTRIB         FIDO_REGISTER_DEF(0xAA09C + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK08ATTRIB         FIDO_REGISTER_DEF(0xAA0A0 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK09ATTRIB         FIDO_REGISTER_DEF(0xAA0A4 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK10ATTRIB         FIDO_REGISTER_DEF(0xAA0A8 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK11ATTRIB         FIDO_REGISTER_DEF(0xAA0AC + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK12ATTRIB         FIDO_REGISTER_DEF(0xAA0B0 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK13ATTRIB         FIDO_REGISTER_DEF(0xAA0B4 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK14ATTRIB         FIDO_REGISTER_DEF(0xAA0B8 + FIDO_MEM_OFFSET)
#define FIDO_MPU_BLK15ATTRIB         FIDO_REGISTER_DEF(0xAA0BC + FIDO_MEM_OFFSET)

#define FIDO_MPU_ATTRIB_SIZE_SHIFT   2
#define FIDO_MPU_ATTRIB_ENABLE       (1 << 0)
#define FIDO_MPU_ATTRIB_READONLY     (1 << 1)

/*  MPU allocation registers */
#define FIDO_CTX0_MPUALLOCATION      FIDO_REGISTER_DEF(0xAA100 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_MPUALLOCATION      FIDO_REGISTER_DEF(0xAA104 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_MPUALLOCATION      FIDO_REGISTER_DEF(0xAA108 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_MPUALLOCATION      FIDO_REGISTER_DEF(0xAA10C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_MPUALLOCATION      FIDO_REGISTER_DEF(0xAA110 + FIDO_MEM_OFFSET)

/*  DCACHE relocatable registers */
#define FIDO_DCACHE_RELOCATBLK00     FIDO_REGISTER_DEF(0xAA180 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK01     FIDO_REGISTER_DEF(0xAA184 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK02     FIDO_REGISTER_DEF(0xAA188 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK03     FIDO_REGISTER_DEF(0xAA18C + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK04     FIDO_REGISTER_DEF(0xAA190 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK05     FIDO_REGISTER_DEF(0xAA194 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK06     FIDO_REGISTER_DEF(0xAA198 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK07     FIDO_REGISTER_DEF(0xAA19C + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK08     FIDO_REGISTER_DEF(0xAA1A0 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK09     FIDO_REGISTER_DEF(0xAA1A4 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK10     FIDO_REGISTER_DEF(0xAA1A8 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK11     FIDO_REGISTER_DEF(0xAA1AC + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK12     FIDO_REGISTER_DEF(0xAA1B0 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK13     FIDO_REGISTER_DEF(0xAA1B4 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK14     FIDO_REGISTER_DEF(0xAA1B8 + FIDO_MEM_OFFSET)
#define FIDO_DCACHE_RELOCATBLK15     FIDO_REGISTER_DEF(0xAA1BC + FIDO_MEM_OFFSET)

/*  Context control registers */
#define FIDO_CTX0_CONTROL            FIDO_REGISTER_DEF(0xAA400 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_CONTROL            FIDO_REGISTER_DEF(0xAA404 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_CONTROL            FIDO_REGISTER_DEF(0xAA408 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_CONTROL            FIDO_REGISTER_DEF(0xAA40C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_CONTROL            FIDO_REGISTER_DEF(0xAA410 + FIDO_MEM_OFFSET)

#define FIDO_CTX_CONTROL_STATE_MASK     ( 3 <<  0)
#define FIDO_CTX_CONTROL_STATE_HALTED   ( 0 <<  0)
#define FIDO_CTX_CONTROL_STATE_NOTREADY ( 1 <<  0)
#define FIDO_CTX_CONTROL_STATE_READY    ( 2 <<  0)

#define FIDO_CTX_CONTROL_PRIORITY_MASK  7
#define FIDO_CTX_CONTROL_PRIORITY_SHIFT 8

#define FIDO_CTX_CONTROL_MODE_MASK      ( 3 << 13)
#define FIDO_CTX_CONTROL_MODE_STANDARD  ( 0 << 13)
#define FIDO_CTX_CONTROL_MODE_FAST      ( 1 << 13)
#define FIDO_CTX_CONTROL_MODE_FAST_ST   ( 3 << 13)

#define FIDO_CTX0_MAXTIME            FIDO_REGISTER_DEF(0xAA480 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_MAXTIME            FIDO_REGISTER_DEF(0xAA484 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_MAXTIME            FIDO_REGISTER_DEF(0xAA488 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_MAXTIME            FIDO_REGISTER_DEF(0xAA48C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_MAXTIME            FIDO_REGISTER_DEF(0xAA490 + FIDO_MEM_OFFSET)

/*  Context timer registers */
#define FIDO_CTX0_TIME               FIDO_REGISTER_DEF(0xAA500 + FIDO_MEM_OFFSET)
#define FIDO_CTX1_TIME               FIDO_REGISTER_DEF(0xAA504 + FIDO_MEM_OFFSET)
#define FIDO_CTX2_TIME               FIDO_REGISTER_DEF(0xAA508 + FIDO_MEM_OFFSET)
#define FIDO_CTX3_TIME               FIDO_REGISTER_DEF(0xAA50C + FIDO_MEM_OFFSET)
#define FIDO_CTX4_TIME               FIDO_REGISTER_DEF(0xAA510 + FIDO_MEM_OFFSET)

#define FIDO_CTXN_TIMECLR            FIDO_REGISTER_DEF(0xAA580 + FIDO_MEM_OFFSET)
#define FIDO_CTX_IDLETIME            FIDO_REGISTER_DEF(0xAA584 + FIDO_MEM_OFFSET)
#define FIDO_CTXN_TIMEEN             FIDO_REGISTER_DEF(0xAA588 + FIDO_MEM_OFFSET)
#define FIDO_CTX_FAULTID             FIDO_REGISTER_DEF(0xAA600 + FIDO_MEM_OFFSET)

/*  Current context register, contains id of currently executing context */
#define FIDO_CURRENT_CONTEXT_REGISTER FIDO_REGISTER_DEF(0xAA604 + FIDO_MEM_OFFSET)

/* UIC Frame Buffer transmit and receive memory */
#define FIDO_FB_XMIT_BASE_ADDR       (0xB0000 + FIDO_MEM_OFFSET)
#define FIDO_FB_XMIT_SIZE            ((1 * 1024) * sizeof (unsigned long))

#define FIDO_FB_RCV_BASE_ADDR        (0xB4000 + FIDO_MEM_OFFSET)
#define FIDO_FB_RCV_SIZE             ((2 * 1024) * sizeof (unsigned long))

/* UIC register file base addresses */
#define FIDO_UIC0_BASE               (0x00B8000 + FIDO_MEM_OFFSET)
#define FIDO_UIC1_BASE               (0x00B8200 + FIDO_MEM_OFFSET)
#define FIDO_UIC2_BASE               (0x00B8400 + FIDO_MEM_OFFSET)
#define FIDO_UIC3_BASE               (0x00B8600 + FIDO_MEM_OFFSET)

#endif /* FIDOmemmap_h */

///@}
